PRO idl_example ; Tested on IDL v8.9

  ; Change these values to point to the product files
  hdf5_file_path = "g3b_ssp_6.0.0_2024100101SR.h5"
  binary_file_path = "g3b_sspb_6.0.0_2024100101SR.dat"

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Plot an ozone profile (altitude vs. concentration) using
  ; a formatted (HDF5) product file
  ; Requires the HDF5 DLM (execute `HELP, 'hdf5', /DLM` for info)

  ; Open the HDF5 file
  file_id = H5F_OPEN(hdf5_file_path)

  ; Isolate the dataset of interest
  dataset_id = H5D_OPEN(file_id,"o3_ao3")
  ; HINT: execute `H5_LIST,hdf5_file_path` to list available datasets

  ; Read the data from the dataset
  ozone_profile = H5D_READ(dataset_id)
  ; Read the fill value from the dataset attributes
  ozone_fill_value = H5A_READ(H5A_OPEN_NAME(dataset_id,"_FillValue"))

  ; For plotting purposes, let's also grab altitudes and event ID using a simpler method
  altitudes = H5_GETDATA(hdf5_file_path,"/altitude")
  event_id = H5_GETDATA(hdf5_file_path,"/event_information/event_id")

  ; Establish a mask for fill values
  mask = WHERE(ozone_profile LT ozone_fill_value)

  ; Plot the ozone profile
  ozone_plot = PLOT(ozone_profile[mask],altitudes[mask],TITLE="Event "+event_id, $
    XTITLE="Ozone Concentration (cm^-3)", YTITLE="Altitude (km)", $
    XRANGE=[0,1e13])

  ; Save the plot
  ozone_plot.SAVE, "ozone_"+event_id+".png"

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ; Plot an aerosol profile (altitude vs. extinction coefficient) using
  ; an unformatted (binary) product file
  ; Requires read_solar.pro

  ; Read the binary product file
  read_solar_v6,binary_file_path,status,data

  ; Aerosol extinction coefficient profiles are given for 9 channels,
  ; lets use the data from the next-to-last (wavelength ~1 micron)
  aerosol_profile = data.aerosol_extinction

  ; Read the fill value for float variables
  aerosol_fill_value = data.float32_fill

  ; For plotting purposes, let's also grab the altitudes (aerosol only
  ; uses the first 90 of these), event ID, and aerosol wavelength
  altitudes = data.altitude
  event_id = data.event_id
  wavelength = data.aerosol_wavelength

  ; Establish a mask for fill values
  mask = WHERE(aerosol_profile LT aerosol_fill_value)

  ; Plot the aerosol extinction profile
  aerosol_plot = PLOT(aerosol_profile[mask],altitudes[mask],TITLE="Event "+event_id, $
    XTITLE=string(wavelength,FORMAT="(F0.1)")+" nm Aerosol Extinction Coefficient (km^-1)", $
    YTITLE="Altitude (km)",/XLOG,XRANGE=[1e-6,1e-2])

  aerosol_plot.SAVE, "aerosol_"+event_id+".png"

END