  ;+
  ; :Description:
  ;    Read in a SAGE III/ISS v6.0 solar transmission binary unformatted file.
  ;
  ; :Params:
  ;    filename - The full file path to the SAGE III/ISS v6.0 binary unformatted file
  ;    status - Returned error status (either 'OK' or 'ERROR')
  ;    solar - An IDL structure variable containing the variables
  ;
  ; :Example:
  ;  read_solar_trans_v6, 'g3b_tb_6.0.0_2017060728SS.dat', status, solar
  ;
  ; :Author: Sean Davis, Michael Heitz (michael.l.heitz@nasa.gov)
  ;-
pro read_solar_trans_v6, filename, status, solar

  ;The header containing the file information (provided by the SAGE team)
  csvfile='index/g3b_tb_6.0.0.csv'
  status='OK'
   
  header=read_csv(csvfile)
  ;Convert the given variable long_names to a string array of valid IDL names
  varname = STRUPCASE( IDL_VALIDNAME(header.field1,/convert_all))
  ;IDL type code for the data types listed in the csv file - these are used when making arrays
  type_table = DICTIONARY('str',1,'int32',3,'float32',4,'bool',1,'float64',5)
     
  ;This will be the structure that is eventually read in from the file
  solar = orderedhash()

  ;Define the data structure variables
  for i = 0, varname.length-1 DO BEGIN
    vardims = FIX(header.field6[i].split(','))     ;field6 contains dimension length, which could be 1D or 2D (e.g., 200,9)
    if (header.field5[i] EQ 'bool') then begin
      arr = make_array(vardims, /BOOLEAN)
    endif else begin
      arr = make_array( vardims, type=type_table[header.field5[i]])
    endelse
    solar[varname[i]] = arr
  ENDFOR

  ;This converts it to a structure in the right order for reading directly
  solar = solar.tostruct()

  on_ioerror, open_error
  openr, lun, filename, /get_lun
  readu, lun, solar
  close, lun
  free_lun, lun

  ;Now we convert it back to a dictionary and then convert the byte array variables to strings and flatten 1-element arrays to scalars
  solar = orderedhash(solar)

  foreach entry, solar, key do BEGIN
    IF entry.length eq 1 THEN BEGIN         ;We made everything an array above using make_array() - this turns 1-element arrays back to scalars
      solar[key] = entry[0]
    ENDIF ELSE BEGIN
      IF ~ISA(entry,'byte',/BOOLEAN) && ISA(entry,'byte') THEN BEGIN      ;If it is **NOT** boolean but IS a byte array, convert it to a string 
        solar[key] = STRING(entry)
      ENDIF
    ENDELSE
  endforeach
                     
  solar = solar.tostruct()
  return

  open_error:
  print, 'Error on open or read of file ', filename
  status = 'ERROR'

END
